'use strict'

entityRegistry['module']['threeDeeCloud'] = {
    extendedInfo: {
        displayName: 'Cloud',
        displayGroup: '3D Effects',
    },
    init: (staticConfig) => {
        const {
            numItems,
            spawnSize,
            ballMass,
            preRoll,
            gravity,
            frameCount,
            seed,
        } = { ...staticConfig }
        const rng = new Math.seedrandom(seed)

        const world = new CANNON.World()
        world.gravity.set(0, 0, gravity)//-9.82) // m/s²

        // Materials
        var groundMaterial = new CANNON.Material('groundMaterial')

        // Adjust constraint equation parameters for ground/ground contact
        var ground_ground_cm = new CANNON.ContactMaterial(groundMaterial, groundMaterial, {
            friction: .8,
            restitution: 0.3,
            contactEquationStiffness: 1e8,
            contactEquationRelaxation: 3,
            frictionEquationStiffness: 1e8,
            frictionEquationRegularizationTime: 3,
        });

        // Add contact material to the world
        world.addContactMaterial(ground_ground_cm)

        const sphereBody = []
        const sphere = new CANNON.Sphere(1)
        for (let i = 0; i < numItems; ++i) {
            const body = new CANNON.Body({
                mass: ballMass,
                position: new CANNON.Vec3((rng()-.5)*spawnSize, (rng()-.5)*spawnSize, (rng()-.5)*spawnSize),
                quaternion: new CANNON.Quaternion(rng(), rng(), rng(), rng()),
                shape: sphere,
                material: groundMaterial
            })
            world.addBody(body)
            // body.applyLocalForce(new CANNON.Vec3(rng()*2-1, rng()*2-1, -(rng()*5+5)), new CANNON.Vec3(rng()*2-1, rng()*2-1, rng()*2-1))
            sphereBody.push(body)
        }

        var fixedTimeStep = 1.0 / 5000.0 // seconds

        for (let i = 0; i < preRoll; ++i) {
            world.step(fixedTimeStep)
        }

        const bodies = []
        for (let i = 0; i < frameCount; ++i) {
            for (let j = 0; j < sphereBody.length; ++j) {
                bodies[j] = bodies[j] || []
                const pos = [sphereBody[j].position.x, -sphereBody[j].position.z, sphereBody[j].position.y]
                const quat = sphereBody[j].quaternion
                const mat = m4.quaternionToMatrix([quat.y, quat.z, quat.x, quat.w])
                bodies[j].push([pos, mat])
            }
            world.step(fixedTimeStep)
        }

        return {
            bodies
        }
    },
    staticConfig: [
        { paramName: 'numItems', displayName: 'Items', type: 'int', defaultValue: 100, triggerInit: true },
        { paramName: 'spawnSize', displayName: 'Spawn Size', type: 'float', defaultValue: 5, triggerInit: true },
        { paramName: 'ballMass', displayName: 'Ball Mass', type: 'float', defaultValue: 5.25, triggerInit: true },
        { paramName: 'preRoll', displayName: 'Pre Roll', type: 'int', defaultValue: 0, triggerInit: true },
        { paramName: 'frameCount', displayName: 'Frame Count', type: 'int', defaultValue: 1000, triggerInit: true },
        { paramName: 'gravity', displayName: 'Gravity', type: 'float', defaultValue: -9.82, triggerInit: true },
        { paramName: 'seed', displayName: 'Seed', type: 'string', defaultValue: 'seed', triggerInit: true },
    ],
    dynamicConfig: [
        { paramName: 'model', displayName: 'Model', type: 'model', defaultValue: '' },
        { paramName: 'itemScale', displayName: 'Item Scale', type: 'float', defaultValue: 1 },
        { paramName: 'animation', displayName: 'Animation', type: 'float', defaultValue: 0 },
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                frameCount,
                model,
                itemScale,
                animation,
            } = { ...config }
            
            const colorBuffer = renderer.getCurrentBuffer('color')
            const depthBuffer = renderer.getCurrentBuffer('depth')
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')

            const scaleMat = m4.scaling(itemScale, itemScale, itemScale)
            const clampedAnimation = clamp(animation, 0, 1)
            const frame = Math.floor(clampedAnimation * (frameCount-1))
            const frameAlpha = (clampedAnimation * (frameCount-1)) % 1
            self.bodies.forEach(body => {
                const pos1 = body[frame][0]
                const pos2 = body[Math.min(frameCount-1, frame+1)][0]
                const pos = m4.lerpVectors(pos1, pos2, frameAlpha)
                if (pos[2] < 12) {
                    const rotationMat = body[frame][1]

                    // const rot1 = body[frame][1]
                    // const rot2 = body[frame+1][1]
                    // const rot = [
                    //     Math.abs(rot1[0]-rot2[0])>Math.PI ? rot1[0] : lerp(rot1[0],rot2[0],frameAlpha),
                    //     Math.abs(rot1[1]-rot2[1])>Math.PI ? rot1[1] : lerp(rot1[1],rot2[1],frameAlpha),
                    //     Math.abs(rot1[2]-rot2[2])>Math.PI ? rot1[2] : lerp(rot1[2],rot2[2],frameAlpha),
                    // ]
                    // const rotationMat = m4.zRotate(m4.yRotate(m4.xRotation(rot[0]), rot[1]), rot[2])

                    const translationMat = m4.translation(pos[0], pos[1], pos[2])
                    const worldMat = m4.multiply(m4.multiply(translationMat, rotationMat), scaleMat)

                    renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
                }
            })
        }
    }
}
